
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 30 10:17:10 2008
   *  Last modified   :  %modify_time%
   */
  /** @file 
      *  \brief This file includes the HMAC ATP tests
   *
   *  \version CRYS_SEP_ATP_HMAC_Acceptance.c#1:csrc:1
   *  \author 
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "CRYS.h"
#include "CRYS_SEP_ATP_Defines.h"
#include "DX_VOS_Mem.h"


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern CRYSError_t      TST_ATP_Error;
/************* Private function prototype ***********************/
/************************ Public Functions ******************************/
CRYSError_t SEPQA_ATP_CheckErrorReturn(CRYSError_t  TST_ErrorRet,
									   char         TST_FuncName[TST_FUNC_STRING_LEN],
									   char        *FailureFunctionNameOut_ptr);
									   

/***************************
* Function Name:                                                           
* 	SEPQA_HMAC_NotIntegrated 		
* 
* Inputs:
* OperationMode - The operation mode.
*
* key_ptr - The pointer to the users key buffer.
*
* keySize - The size of the received key.
* 
* DataIn_ptr - The pointer to the buffer of the input data to the HMAC. The pointer does 
*                         not need to be aligned. On CSI input mode the pointer must be equal to
*                         value (0xFFFFFFFC | DataInAlignment). 
* 
* DataInSize - The size of the data to be hashed in bytes. On CSI data transfer mode the size must  
*                         multiple of HASH_BLOCK_SIZE for used HASH mode.
*
* DataExpSize - The size of the expected data
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
*
* Outputs:
* 	CRYSError_t - Function error return		    					  		
*
* Description:																	
* 	This function Is a Acceptance test for HMAC non integrated API's							
* 	This function HMAC known vectors using SHA1\SHA224\SHA256\SHA384\SHA512 and MD5
* 	Vector sizes used (in bytes) - 16/32/512/1536									
* Algorithm:	
* 1. Allocate memory buffers
* 2. Call the CRYS_HMAC_Init function to start the encryption/decryption process
* 3. Call the CRYS_HMAC_Update - This function is called N times
*                                (N = number of blocks in data each block is 16byte)
* 4. Call the CRYS_HMAC_Finish to end the HMAC process and get the Digest result
* 5. Compare the outputted result to the expected one													
* 6. Free allocated buffers
*
**************************/
CRYSError_t SEPQA_HMAC_NotIntegrated(CRYS_HASH_OperationMode_t   OperationMode,
									 DxUint8_t                  *Key_ptr,
									 DxUint16_t                  KeySize,
									 DxUint8_t                  *DataIn_ptr,
									 DxUint32_t                  DataInSize,
									 DxUint8_t                  *ExpectedData_ptr,        
									 DxUint32_t                  DataExpSize,
									 char                       *FailureFunctionNameOut_ptr)
									 
{
	/*** TST Variables Definitions ****/
	CRYSError_t             	TST_TestStatus=0;
	DxUint32_t              	TST_LastBlockSize;
	CRYS_HMACUserContext_t  	TST_ContextID_ptr;
	DxUint32_t					Block_index = 0;
	DxUint32_t					NumOfBlocks = 0;
	DxUint8_t					*pDataIn;
	DxUint8_t					*pDataOut;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(CRYS_HASH_RESULT_SIZE_IN_WORDS*sizeof(DxUint32_t));

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/* 2. calling the HMAC init function */
	/*************************************/
	TST_ATP_Error = CRYS_HMAC_Init(&TST_ContextID_ptr,
		OperationMode,
		Key_ptr,
		KeySize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_HMAC_Init",
		FailureFunctionNameOut_ptr);


	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 3. calling the HMAC Update function */
	/***************************************/
	NumOfBlocks = DataInSize/TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS;
	if (DataInSize % TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS != 0)
		NumOfBlocks++;

	TST_LastBlockSize = (DxUint32_t)(DataInSize - ((NumOfBlocks-1)*TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS));

	for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*HMAC block loop*/
	{
		if (Block_index != NumOfBlocks-1)
		{
			TST_ATP_Error = CRYS_HMAC_Update(&TST_ContextID_ptr,
				pDataIn+(TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS*Block_index),
				TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));
		}
		else /*Handling if last block size is smaller then TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS*/
		{
			TST_ATP_Error = CRYS_HMAC_Update(&TST_ContextID_ptr,
				pDataIn+(TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS*Block_index),
				TST_LastBlockSize);
		}

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_HMAC_Update",
			FailureFunctionNameOut_ptr);        
			

		if (TST_TestStatus != TESTPASS) 
			goto End;
	}/*End HMAC block loop*/


	/* 4. calling the Finish function */
	/**********************************/
	TST_ATP_Error = CRYS_HMAC_Finish(&TST_ContextID_ptr,
									(DxUint32_t*)pDataOut);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_HMAC_Finish",
		FailureFunctionNameOut_ptr);       
		

	if (TST_TestStatus != TESTPASS) 
		goto End;


	/* 5. Compare the result to expected result */
	/********************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
		ExpectedData_ptr,
		DataExpSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut_ptr);      
		

End:
	/* 6. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;	
};

 /***************************
 * Function Name:                                                           
 * 	SEPQA_HMAC_Integrated 		
 * 
 * Inputs:
 * OperationMode - The operation mode.
 *
 * key_ptr - The pointer to the users key buffer.
 *
 * keySize - The size of the received key.
 * 
 * DataIn_ptr - The pointer to the buffer of the input data to the HMAC. The pointer does 
 *                         not need to be aligned. On CSI input mode the pointer must be equal to
 *                         value (0xFFFFFFFC | DataInAlignment). 
 * 
 * DataInSize - The size of the data to be hashed in bytes. On CSI data transfer mode the size must  
 *                         multiple of HASH_BLOCK_SIZE for used HASH mode.
 *
 * DataExpSize - The size of the expected data
 * FailureFunctionNameOut_ptr - In case that function failures this parameter
 *                              returns the name of the function that failures
 *
 * Outputs:
 * 	CRYSError_t - Function error return		    					  		
 *
 * Description:																	
 * 	This function Is a Acceptance test for HMAC integrated API's							
 * 	This function HMAC known vectors using SHA1\SHA224\SHA256\SHA384\SHA512 and MD5
 * 	Vector sizes used (in bytes) - 16/32/512/1536									
 * Algorithm:	
 * 1. Allocate memory buffers
 * 2. Call the CRYS_HMAC function to execute the HMAC process
 * 3. Compare the outputted result to the expected one													
 * 4. Free allocated buffers
 *
 **************************/
 CRYSError_t SEPQA_HMAC_Integrated(CRYS_HASH_OperationMode_t   OperationMode,
									 DxUint8_t                  *Key_ptr,
									 DxUint16_t                  KeySize,
									 DxUint8_t                  *DataIn_ptr,
									 DxUint32_t                  DataInSize,
									 DxUint8_t                  *ExpectedData_ptr,        
									 DxUint32_t                  DataExpSize,
									 char                       *FailureFunctionNameOut_ptr)									 
 {
	 /*** TST Variables Definitions ****/
	 CRYSError_t             	TST_TestStatus=0;
	 DxUint8_t					*pDataIn;
	 DxUint8_t					*pDataOut;

	 /* 1. Allocate memory buffers */
	 /******************************/
	 pDataIn = DX_VOS_MemMalloc(DataInSize);
	 pDataOut = DX_VOS_MemMalloc(CRYS_HASH_RESULT_SIZE_IN_WORDS*sizeof(DxUint32_t));

	 if ((pDataIn == DX_NULL) ||
		 (pDataOut == DX_NULL))
	 {

		 TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			 "DX_VOS_MemMalloc",
			 FailureFunctionNameOut_ptr);

		 goto End;                                                                       
	 }

	 DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	 /* 2. calling the HMAC function */
	 /*****************************/
	 TST_ATP_Error = CRYS_HMAC(OperationMode,
		 Key_ptr,
		 KeySize,
		 pDataIn,
		 DataInSize,
		 (DxUint32_t*)pDataOut);

	 TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		 "CRYS_HMAC",
		 FailureFunctionNameOut_ptr);       
		 

	 if (TST_TestStatus != TESTPASS) 
		 goto End;


	 /* 3. compare the output data with expected data */
	 /*************************************************/
	 TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
		 ExpectedData_ptr,
		 DataExpSize);

	 TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		 "DX_VOS_MemCmp",
		 FailureFunctionNameOut_ptr);       
		 

End:
	 /* 4. Free allocated buffers */
	 /*****************************/
	 DX_VOS_MemFree(pDataIn);
	 DX_VOS_MemFree(pDataOut);

	 return TST_TestStatus;	
 }










